"use strict";
'use client';

var _interopRequireWildcard = require("@babel/runtime/helpers/interopRequireWildcard").default;
var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault").default;
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useGridPaginationModel = exports.getDerivedPaginationModel = void 0;
var _extends2 = _interopRequireDefault(require("@babel/runtime/helpers/extends"));
var React = _interopRequireWildcard(require("react"));
var _isDeepEqual = require("@mui/x-internals/isDeepEqual");
var _gridFilterSelector = require("../filter/gridFilterSelector");
var _density = require("../density");
var _utils = require("../../utils");
var _pipeProcessing = require("../../core/pipeProcessing");
var _gridPaginationSelector = require("./gridPaginationSelector");
var _gridPaginationUtils = require("./gridPaginationUtils");
const getDerivedPaginationModel = (paginationState, signature, paginationModelProp) => {
  let paginationModel = paginationState.paginationModel;
  const rowCount = paginationState.rowCount;
  const pageSize = paginationModelProp?.pageSize ?? paginationModel.pageSize;
  const page = paginationModelProp?.page ?? paginationModel.page;
  const pageCount = (0, _gridPaginationUtils.getPageCount)(rowCount, pageSize, page);
  if (paginationModelProp && (paginationModelProp?.page !== paginationModel.page || paginationModelProp?.pageSize !== paginationModel.pageSize)) {
    paginationModel = paginationModelProp;
  }
  const validPage = pageSize === -1 ? 0 : (0, _gridPaginationUtils.getValidPage)(paginationModel.page, pageCount);
  if (validPage !== paginationModel.page) {
    paginationModel = (0, _extends2.default)({}, paginationModel, {
      page: validPage
    });
  }
  (0, _gridPaginationUtils.throwIfPageSizeExceedsTheLimit)(paginationModel.pageSize, signature);
  return paginationModel;
};

/**
 * @requires useGridFilter (state)
 * @requires useGridDimensions (event) - can be after
 */
exports.getDerivedPaginationModel = getDerivedPaginationModel;
const useGridPaginationModel = (apiRef, props) => {
  const logger = (0, _utils.useGridLogger)(apiRef, 'useGridPaginationModel');
  const densityFactor = (0, _utils.useGridSelector)(apiRef, _density.gridDensityFactorSelector);
  const previousFilterModel = React.useRef((0, _gridFilterSelector.gridFilterModelSelector)(apiRef));
  const rowHeight = Math.floor(props.rowHeight * densityFactor);
  apiRef.current.registerControlState({
    stateId: 'paginationModel',
    propModel: props.paginationModel,
    propOnChange: props.onPaginationModelChange,
    stateSelector: _gridPaginationSelector.gridPaginationModelSelector,
    changeEvent: 'paginationModelChange'
  });

  /**
   * API METHODS
   */
  const setPage = React.useCallback(page => {
    const currentModel = (0, _gridPaginationSelector.gridPaginationModelSelector)(apiRef);
    if (page === currentModel.page) {
      return;
    }
    logger.debug(`Setting page to ${page}`);
    apiRef.current.setPaginationModel({
      page,
      pageSize: currentModel.pageSize
    });
  }, [apiRef, logger]);
  const setPageSize = React.useCallback(pageSize => {
    const currentModel = (0, _gridPaginationSelector.gridPaginationModelSelector)(apiRef);
    if (pageSize === currentModel.pageSize) {
      return;
    }
    logger.debug(`Setting page size to ${pageSize}`);
    apiRef.current.setPaginationModel({
      pageSize,
      page: currentModel.page
    });
  }, [apiRef, logger]);
  const setPaginationModel = React.useCallback(paginationModel => {
    const currentModel = (0, _gridPaginationSelector.gridPaginationModelSelector)(apiRef);
    if (paginationModel === currentModel) {
      return;
    }
    logger.debug("Setting 'paginationModel' to", paginationModel);
    apiRef.current.setState(state => (0, _extends2.default)({}, state, {
      pagination: (0, _extends2.default)({}, state.pagination, {
        paginationModel: getDerivedPaginationModel(state.pagination, props.signature, paginationModel)
      })
    }), 'setPaginationModel');
  }, [apiRef, logger, props.signature]);
  const paginationModelApi = {
    setPage,
    setPageSize,
    setPaginationModel
  };
  (0, _utils.useGridApiMethod)(apiRef, paginationModelApi, 'public');

  /**
   * PRE-PROCESSING
   */
  const stateExportPreProcessing = React.useCallback((prevState, context) => {
    const paginationModel = (0, _gridPaginationSelector.gridPaginationModelSelector)(apiRef);
    const shouldExportPaginationModel =
    // Always export if the `exportOnlyDirtyModels` property is not activated
    !context.exportOnlyDirtyModels ||
    // Always export if the `paginationModel` is controlled
    props.paginationModel != null ||
    // Always export if the `paginationModel` has been initialized
    props.initialState?.pagination?.paginationModel != null ||
    // Export if `page` or `pageSize` is not equal to the default value
    paginationModel.page !== 0 && paginationModel.pageSize !== (0, _gridPaginationUtils.defaultPageSize)(props.autoPageSize);
    if (!shouldExportPaginationModel) {
      return prevState;
    }
    return (0, _extends2.default)({}, prevState, {
      pagination: (0, _extends2.default)({}, prevState.pagination, {
        paginationModel
      })
    });
  }, [apiRef, props.paginationModel, props.initialState?.pagination?.paginationModel, props.autoPageSize]);
  const stateRestorePreProcessing = React.useCallback((params, context) => {
    const paginationModel = context.stateToRestore.pagination?.paginationModel ? (0, _extends2.default)({}, (0, _gridPaginationUtils.getDefaultGridPaginationModel)(props.autoPageSize), context.stateToRestore.pagination?.paginationModel) : (0, _gridPaginationSelector.gridPaginationModelSelector)(apiRef);
    apiRef.current.setState(state => (0, _extends2.default)({}, state, {
      pagination: (0, _extends2.default)({}, state.pagination, {
        paginationModel: getDerivedPaginationModel(state.pagination, props.signature, paginationModel)
      })
    }), 'stateRestorePreProcessing');
    return params;
  }, [apiRef, props.autoPageSize, props.signature]);
  (0, _pipeProcessing.useGridRegisterPipeProcessor)(apiRef, 'exportState', stateExportPreProcessing);
  (0, _pipeProcessing.useGridRegisterPipeProcessor)(apiRef, 'restoreState', stateRestorePreProcessing);

  /**
   * EVENTS
   */
  const handlePaginationModelChange = () => {
    const paginationModel = (0, _gridPaginationSelector.gridPaginationModelSelector)(apiRef);
    if (apiRef.current.virtualScrollerRef?.current) {
      apiRef.current.scrollToIndexes({
        rowIndex: paginationModel.page * paginationModel.pageSize
      });
    }
  };
  const handleUpdateAutoPageSize = React.useCallback(() => {
    if (!props.autoPageSize) {
      return;
    }
    const dimensions = apiRef.current.getRootDimensions();
    const maximumPageSizeWithoutScrollBar = Math.max(1, Math.floor(dimensions.viewportInnerSize.height / rowHeight));
    apiRef.current.setPageSize(maximumPageSizeWithoutScrollBar);
  }, [apiRef, props.autoPageSize, rowHeight]);
  const handleRowCountChange = React.useCallback(newRowCount => {
    if (newRowCount == null) {
      return;
    }
    const paginationModel = (0, _gridPaginationSelector.gridPaginationModelSelector)(apiRef);
    if (paginationModel.page === 0) {
      return;
    }
    const pageCount = (0, _gridPaginationSelector.gridPageCountSelector)(apiRef);
    if (paginationModel.page > pageCount - 1) {
      apiRef.current.setPage(Math.max(0, pageCount - 1));
    }
  }, [apiRef]);

  /**
   * Goes to the first row of the grid
   */
  const navigateToStart = React.useCallback(() => {
    const paginationModel = (0, _gridPaginationSelector.gridPaginationModelSelector)(apiRef);
    if (paginationModel.page !== 0) {
      apiRef.current.setPage(0);
    }

    // If the page was not changed it might be needed to scroll to the top
    const scrollPosition = apiRef.current.getScrollPosition();
    if (scrollPosition.top !== 0) {
      apiRef.current.scroll({
        top: 0
      });
    }
  }, [apiRef]);

  /**
   * Resets the page only if the active items or quick filter has changed from the last time.
   * This is to avoid resetting the page when the filter model is changed
   * because of and update of the operator from an item that does not have the value
   * or reseting when the filter panel is just opened
   */
  const handleFilterModelChange = React.useCallback(filterModel => {
    const currentActiveFilters = (0, _extends2.default)({}, filterModel, {
      // replace items with the active items
      items: (0, _gridFilterSelector.gridFilterActiveItemsSelector)(apiRef)
    });
    if ((0, _isDeepEqual.isDeepEqual)(currentActiveFilters, previousFilterModel.current)) {
      return;
    }
    previousFilterModel.current = currentActiveFilters;
    navigateToStart();
  }, [apiRef, navigateToStart]);
  (0, _utils.useGridEvent)(apiRef, 'viewportInnerSizeChange', handleUpdateAutoPageSize);
  (0, _utils.useGridEvent)(apiRef, 'paginationModelChange', handlePaginationModelChange);
  (0, _utils.useGridEvent)(apiRef, 'rowCountChange', handleRowCountChange);
  (0, _utils.useGridEvent)(apiRef, 'sortModelChange', navigateToStart);
  (0, _utils.useGridEvent)(apiRef, 'filterModelChange', handleFilterModelChange);

  /**
   * EFFECTS
   */
  const isFirstRender = React.useRef(true);
  React.useEffect(() => {
    if (isFirstRender.current) {
      isFirstRender.current = false;
      return;
    }
    if (!props.pagination) {
      return;
    }
    apiRef.current.setState(state => (0, _extends2.default)({}, state, {
      pagination: (0, _extends2.default)({}, state.pagination, {
        paginationModel: getDerivedPaginationModel(state.pagination, props.signature, props.paginationModel)
      })
    }));
  }, [apiRef, props.paginationModel, props.signature, props.pagination]);
  React.useEffect(() => {
    apiRef.current.setState(state => {
      const isEnabled = props.pagination === true;
      if (state.pagination.paginationMode === props.paginationMode || state.pagination.enabled === isEnabled) {
        return state;
      }
      return (0, _extends2.default)({}, state, {
        pagination: (0, _extends2.default)({}, state.pagination, {
          paginationMode: props.paginationMode,
          enabled: props.pagination === true
        })
      });
    });
  }, [apiRef, props.paginationMode, props.pagination]);
  React.useEffect(handleUpdateAutoPageSize, [handleUpdateAutoPageSize]);
};
exports.useGridPaginationModel = useGridPaginationModel;